#!/bin/bash
# vim: tabstop=3 
#
# 	A cron job script for downloading custom node announcements with error reporting 
#
# 	This is intended to be run from cron every 8 hours and will randomly insert 
#	a delay from 0-8 hours
#
#  Copyright (C) 2020 Bryan Fields
#	bryan@bryanfields.net
#
#	This program is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; either version 2 of the License, or
#	(at your option) any later version.
#
#	This program is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License along
#	with this program; if not, write to the Free Software Foundation, Inc.,
#	51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
# 
#--------------------------- Revision History ----------------------------------
#	2018-12-22	bfields	Release of 1.04
#  2020-12-31	bfields	1.05 - changed allstarlink.org to pttlink.org due to perfidy

#DEFAULTS
URL="rsync://rsync.pttlink.org/connect-messages"
FILEPATH="/var/lib/asterisk/sounds/rpt/nodenames"
DEBUG="NO"
DEFAULT_INTERVAL="28800"
NOW="NO"
STD_ERROR_LOG="NO"
HELP="NO"
VERSION="1.05"
COPYRIGHT="Copyright 2020 Bryan Fields"

#Parse CLI arguments
for i in "$@"
do
case $i in
	-d|--debug)
	DEBUG="YES"
	shift # past argument=value
	;;
 	-u=*|--url=*)
 	URL="${i#*=}"
	shift # past argument=value
	;;
	-f=*|--filepath=*)
	FILEPATH="${i#*=}"
	shift # past argument=value
	;;
	-n|--now)
	NOW='YES'
	shift # past argument with no value
	;;
	-i=*|--interval=*)
	OPTINTERVAL="${i#*=}"
	shift # past argument=value
	;;
	-h|--help)
	HELP='YES'
	shift # past argument with no value
	;;
	-s|--stdErrorEnable)
	STD_ERROR_LOG='YES'
	shift # past argument with no value
	;;
	*)  # unknown option
	;;
esac
done

if [[ -n $1 ]]; then
	echo "Invalid argument: $1"
	exit
fi



#set INTERVAL if not set on CLI
if [ -z "$OPTINTERVAL" ]; then
	INTERVAL="$DEFAULT_INTERVAL"
elif [ -n "$OPTINTERVAL" ]; then
	INTERVAL="$OPTINTERVAL"
fi

#test if NOW and INTEVAL are set
if [ $NOW == "YES" ] && [ -n "$OPTINTERVAL" ]; then
	echo "NodeNameUpdate ERROR: Cannot set interval and NOW"
	exit 2
fi

HELPTEXT="
NodeNameUpdate.sh - Custom node name updater for AllStar Link nodes.
Version: $VERSION
$COPYRIGHT

This script is meant to be run from contab three times per day. It will sleep
for a random interval (eight hours by default). 

Please see this wiki for more info:
https://wiki.pttlink.org/wiki/Custom_Node_Announcements

Command Line Options:
-u= --url=      : Set the URL used for rsync
                : Default <$URL>
                : 
-f= --filepath= : Set the local file path for downloaded names
                : Default $FILEPATH
                : 
-i= --interval= : Interval used for sleeping of script in seconds
                : Default $DEFAULT_INTERVAL seconds for sleeping
                : This should match the interval used in crontab
                : -n is incompatible with this flag
                : 
-s --stdErrEn   : Enable StdErr output upon successful runs
                : by default this is disabled and only will log to syslog
                : failures will still log to StdErr
                : 
Interactive shell only options:
-d --debug      : Set debug flags
-n --now        : Run now, don't sleep before running
-h --help       : Show this help.

Typical usage:
Crontab:
0 0,8,16 * * *  root /usr/local/bin/NodeNameUpdate.sh
"
if [ $HELP = 'YES' ] ; then
   echo "$HELPTEXT"
   exit 0
fi


#set sleep for seconds
if [ $NOW = 'NO' ] ; then
	SLEEP=$((( $RANDOM % $INTERVAL ) + 1))
	if [ $DEBUG = 'YES' ] ; then
		echo "Sleeping for $SLEEP SECONDS"
	fi
	#sleep for a random time of 0-28800 seconds (8 hours)
	sleep $SLEEP"s"
fi

# print vars if DEBUG is set
if [ $DEBUG = 'YES' ] ; then
	echo -e "URL"'\t'" = $URL"
	echo -e "FilePath = $FILEPATH"
fi

#IF sedug is set, then use the below to run rync
if [ $DEBUG = 'YES' ] ; then
	rsync -a --stats $URL $FILEPATH
	exit 0
fi

if [ $DEBUG != 'YES' ] ; then

	#assign the filesUpdated var with the number of files changed by rsync.  This is nasty, but it works.

	filesUpdated=`rsync -a --stats $URL $FILEPATH |grep "Number of regular files transferred" | cut -c 38-50`
	exitcode=$?
	if [ "$exitcode" = "0" ]; then
    	if [ "$filesUpdated" = "0" ]; then
			if [ "$STD_ERROR_LOG" != "NO" ]; then
				echo "NodeNameUpdate run with no updates"
			fi
			logger -i "NodeNameUpdate run with no updates"
    	elif [ "$filesUpdated" > "0" ]; then
			if [ "$STD_ERROR_LOG" != "NO" ]; then 
				echo "NodeNameUpdate updated $filesUpdated nodenames"
			fi
        	logger -i "NodeNameUpdate updated $filesUpdated nodenames" 
    fi
	else
   	echo "NodeNameUpdate failed"
   	logger -i "NodeUpdate failed"
    	exit 1
	fi
fi
exit 0
