#!/bin/bash
# vim: tabstop=3 
#
# 	A cron job script for downloading the extnodes file with error reporting
# 
#
# 	This is intended to be run from cron every 5 min
#
#
#  Copyright (C) 2020 Bryan Fields
#  bryan@bryanfields.net
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#  
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License along
#  with this program; if not, write to the Free Software Foundation, Inc.,
#  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
# 
#--------------------------- Revision History ----------------------------------
#	2018-12-22	bfields	Release of 1.02
#  2020-12-31	bfields	1.05 - changed allstarlink.org to pttlink.org due to perfidy



URL="http://nodes.regsvcs.pttlink.org/cgi-bin/gennodes.php"
FILEPATH=/var/lib/asterisk
DEBUG="NO"
DEFAULT_INTERVAL="300"
NOW="NO"
STD_ERROR_LOG="NO"
HELP="NO"
VERSION="1.05"
COPYRIGHT="Copyright 2020 Bryan Fields"

#Parse CLI arguments
for i in "$@"
do
case $i in
	-d|--debug)
	DEBUG="YES"
	shift # past argument=value
	;;
 	-u=*|--url=*)
 	URL="${i#*=}"
	shift # past argument=value
	;;
	-f=*|--filepath=*)
	FILEPATH="${i#*=}"
	shift # past argument=value
	;;
	-n|--now)
	NOW='YES'
	shift # past argument with no value
	;;
	-i=*|--interval=*)
	OPTINTERVAL="${i#*=}"
	shift # past argument=value
	;;
	-h|--help)
	HELP='YES'
	shift # past argument with no value
	;;
	-s|--stdErrorEnable)
	STD_ERROR_LOG='YES'
	shift # past argument with no value
	;;
	*)  # unknown option
	;;
esac
done

if [[ -n $1 ]]; then
	echo "Invalid argument: $1"
	exit
fi



#set INTERVAL if not set on CLI
if [ -z "$OPTINTERVAL" ]; then
	INTERVAL="$DEFAULT_INTERVAL"
elif [ -n "$OPTINTERVAL" ]; then
	INTERVAL="$OPTINTERVAL"
fi

#test if NOW and INTEVAL are set
if [ $NOW == "YES" ] && [ -n "$OPTINTERVAL" ]; then
	echo "NodeNameUpdate ERROR: Cannot set interval and NOW"
	exit 2
fi

HELPTEXT="
NodeUpdate.sh - AllStar Link node list updater.
Version: $VERSION
$COPYRIGHT

This script is meant to be run from contab every five minutes. It will sleep
for a random interval (five minutes by default) before updating the node list.
This random delay ensures nodes don't hit the servers all at the same time. 

Please see this wiki for more info:
https://wiki.pttlink.org/wiki/NodeUpdate

Command Line Options:
-u= --url=      : Set the URL used for nodes list
                : Default <$URL>
                : 
-f= --filepath= : Set the local file path for downloaded names
                : Default $FILEPATH
                : 
-i= --interval= : Interval used for sleeping of script in seconds
                : Default $DEFAULT_INTERVAL seconds for sleeping
                : This should match the interval used in crontab
                : -n is incompatible with this flag
                : 
-s --stdErrEn   : Enable StdErr output upon successful runs
                : by default this is disabled and only will log to syslog
                : failures will still log to StdErr
                : 
Interactive shell only options:
-d --debug      : Set debug flags
-n --now        : Run now, don't sleep before running
-h --help       : Show this help.

Typical usage:
Crontab:
0/5 * * * *  root /usr/local/bin/NodeUpdate.sh
"


if [ $HELP = 'YES' ] ; then
   echo "$HELPTEXT"
   exit 0
fi
#set sleep for seconds
if [ $NOW = 'NO' ] ; then
	SLEEP=$((( $RANDOM % $INTERVAL ) + 1))
	if [ $DEBUG = 'YES' ] ; then
		echo "Sleeping for $SLEEP SECONDS"
	fi
	#sleep for a random time 
	sleep $SLEEP"s"
fi

# print vars if DEBUG is set
if [ $DEBUG = 'YES' ] ; then
	echo -e "URL"'\t'" = $URL"
	echo -e "FilePath = $FILEPATH"
fi

if [ $DEBUG = 'YES' ] ; then
	curl --compressed --connect-timeout 5 -f -m 20 -o /tmp/rpt_extnodes-temp $URL
	exitcode=$?
elif [ $DEBUG != 'YES' ] ; then
	# use curl in silent and compressed mode, connect timeout of 5 and
	# -f for fail with an error code of 22 on server errors
	#-m 20 is a max time for the whole process of 20 seconds
	curl -s --compressed --connect-timeout 5 -f -m 20 -o /tmp/rpt_extnodes-temp $URL
	exitcode=$?
fi
if [ "$exitcode" = "0" ]; then
	grep -q extnodes /tmp/rpt_extnodes-temp
	if [ $? -eq "0" ] ;then
		chmod 700 /tmp/rpt_extnodes-temp
		cp /tmp/rpt_extnodes-temp $FILEPATH/rpt_extnodes-temp
		mv -f $FILEPATH/rpt_extnodes-temp $FILEPATH/rpt_extnodes
		whenandwhere=`grep "Generated at" /tmp/rpt_extnodes-temp |cut -c2-100`
		if [ $DEBUG = 'YES' ] || [ $STD_ERROR_LOG = 'YES' ] ; then
			#Only output if we have dubg enabled 
				echo "DEBUG: NodeUpdate installed a new nodes list $whenandwhere"
		fi
		logger -i "NodeUpdate installed a new nodes list $whenandwhere"
		else
		if [ $DEBUG = 'YES' ] || [ $STD_ERROR_LOG = 'YES' ]; then
			echo "ERROR: NodeUpdate failed to parse node list; node list not updated"
		fi
		logger -i "ERROR: NodeUpdate failed to parse node list; node list not updated"
		exit 1
	fi
else
	if [ $DEBUG = 'YES' ] || [ $STD_ERROR_LOG = 'YES' ]; then
		echo "ERROR: NodeUpdate could not get nodes list from server"
	fi
	logger -i "NodeUpdate could not get nodes list from server"
	exit 1
fi
exit 0
